/**
 * v4l2splash.c
 * based  mx6-vout.c
 *
 * i.MX6 V4L2 Video Output Overlay device tester.
 *
 * by
 *
 * Steve Longerbeam, <steve_longerbeam@mentor.com>
 * Copyright (C) 2012 Mentor Graphics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdint.h>
#include <signal.h>
#include <stdbool.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <linux/videodev2.h>

#include "drm-mode-set-helper.h"

#define perror_exit(cond, func)					\
	if (cond) {						\
		fprintf(stderr, "%s:%d: ", __func__, __LINE__);	\
		perror(func);					\
		exit(EXIT_FAILURE);				\
	}

#define error_exit(cond, func)						\
	if (cond) {							\
		fprintf(stderr, "%s:%d: failed\n", func, __LINE__);	\
		exit(EXIT_FAILURE);					\
	}

#define perror_ret(cond, func)					\
	if (cond) {						\
		fprintf(stderr, "%s:%d: ", __func__, __LINE__);	\
		perror(func);					\
		return ret;					\
	}

#define memzero(x)				\
	memset(&(x), 0, sizeof (x));

#define PROCESS_DEBUG 1
#ifdef PROCESS_DEBUG
#define debug(msg, ...)						\
	fprintf(stderr, "%s: " msg, __func__, ##__VA_ARGS__);
#else
#define debug(msg, ...)
#endif

//#define PROCESS_VDEBUG 1
#ifdef PROCESS_VDEBUG
#define vdebug(msg, ...)					\
	fprintf(stderr, "%s: " msg, __func__, ##__VA_ARGS__);
#else
#define vdebug(msg, ...)
#endif

#define MAX_NUM_BUFFERS 8

static int cap_fd, vout_fd;
static char *p_cap_mmap[MAX_NUM_BUFFERS];
static size_t cap_frame_size[MAX_NUM_BUFFERS];
static unsigned int num_vout_frames = 0;
static unsigned int num_cap_frames = 0;

static char *p_vout_frame[MAX_NUM_BUFFERS];
static size_t vout_frame_size[MAX_NUM_BUFFERS];

unsigned char *p_capture_fb = NULL;

/* Command-line params */

#define DEF_CAP_DEV_NAME  "/dev/video0"
#define DEF_VOUT_DEV_NAME "/dev/video3"
#define DEF_CAP_INPUT    0
#define DEF_CAP_WIDTH  640
#define DEF_CAP_MOTION   0
#define DEF_CAP_HEIGHT 480
#define DEF_DST_WIDTH  1024
#define DEF_DST_HEIGHT 768
#define DEF_DST_X        0
#define DEF_DST_Y        0
#define DEF_CAP_PIXFMT "BGR3" // V4L2_PIX_FMT_BGR24
#define DEF_CAP_FR      30
#define DEF_DST_PIXFMT "RGBP" // V4L2_PIX_FMT_RGB565
#define DEF_VOUT_MEMTYPE "dmabuf"
#define DEF_CAP_ROTATION 0
#define DEF_DST_ROTATION 0
#define DEF_NUM_BUFFERS  1
#define DEF_BMP_SIZE_OFFSET    18
#define DEF_BMP_BPP_OFFSET     28
#define DEF_BMP_RASTER_OFFSET  10
#define DEF_SPLASH_TIMEOUT 3000
#define DEF_CAMERA_TIMEOUT 3000
#define DEF_CONNECTOR_NAME "VGA"
#define DEF_DRI_CARD "/dev/dri/card0"

static char *cap_dev_name      = DEF_CAP_DEV_NAME;
static char *vout_dev_name     = DEF_VOUT_DEV_NAME;
static unsigned int cap_width  = DEF_CAP_WIDTH;
static unsigned int cap_height = DEF_CAP_HEIGHT;
static int cap_input           = DEF_CAP_INPUT;
static int cap_motion          = DEF_CAP_MOTION;
static char *cap_pixfmt        = DEF_CAP_PIXFMT;
static int cap_fr              = DEF_CAP_FR;
static int dst_width           = DEF_DST_WIDTH;
static int dst_height          = DEF_DST_HEIGHT;
static int dst_x               = DEF_DST_X;
static int dst_y               = DEF_DST_Y;
static char *dst_pixfmt        = DEF_DST_PIXFMT;
static int cap_rotation        = DEF_CAP_ROTATION;
static int dst_rotation        = DEF_DST_ROTATION;
static int splash_timeout      = DEF_SPLASH_TIMEOUT;
static int camera_timeout      = DEF_CAMERA_TIMEOUT;
static bool cap_flip_h, cap_flip_v;
static bool dst_flip_h, dst_flip_v;
static char *cap_frame_file[MAX_NUM_BUFFERS];
static int num_buffers         = DEF_NUM_BUFFERS;
static bool enable_galpha = false;
static bool enable_lalpha = false;
static uint8_t galpha = 255;
static bool enable_ck = false;
static bool is_bmp = false;
static uint32_t color_key = 0;
struct v4l2_crop cap_crop;
static bool use_key = false;
static char *connector_name = DEF_CONNECTOR_NAME;
static char *dri_card = DEF_DRI_CARD;

static const char * d2s_std_id(int v)
{
    switch (v) {
    case V4L2_STD_PAL_B: return "V4L2_STD_PAL_B";
    case V4L2_STD_PAL_B1: return "V4L2_STD_PAL_B1";
    case V4L2_STD_PAL_G: return "V4L2_STD_PAL_G";
    case V4L2_STD_PAL_H: return "V4L2_STD_PAL_H";
    case V4L2_STD_PAL_I: return "V4L2_STD_PAL_I";
    case V4L2_STD_PAL_D: return "V4L2_STD_PAL_D";
    case V4L2_STD_PAL_D1: return "V4L2_STD_PAL_D1";
    case V4L2_STD_PAL_K: return "V4L2_STD_PAL_K";
    case V4L2_STD_PAL_M: return "V4L2_STD_PAL_M";
    case V4L2_STD_PAL_N: return "V4L2_STD_PAL_N";
    case V4L2_STD_PAL_Nc: return "V4L2_STD_PAL_Nc";
    case V4L2_STD_PAL_60: return "V4L2_STD_PAL_60";
    case V4L2_STD_NTSC_M: return "V4L2_STD_NTSC_M";
    case V4L2_STD_NTSC_M_JP: return "V4L2_STD_NTSC_M_JP";
    case V4L2_STD_NTSC_443: return "V4L2_STD_NTSC_443";
    case V4L2_STD_NTSC_M_KR: return "V4L2_STD_NTSC_M_KR";
    case V4L2_STD_SECAM_B: return "V4L2_STD_SECAM_B";
    case V4L2_STD_SECAM_D: return "V4L2_STD_SECAM_D";
    case V4L2_STD_SECAM_G: return "V4L2_STD_SECAM_G";
    case V4L2_STD_SECAM_H: return "V4L2_STD_SECAM_H";
    case V4L2_STD_SECAM_K: return "V4L2_STD_SECAM_K";
    case V4L2_STD_SECAM_K1: return "V4L2_STD_SECAM_K1";
    case V4L2_STD_SECAM_L: return "V4L2_STD_SECAM_L";
    case V4L2_STD_SECAM_LC: return "V4L2_STD_SECAM_LC";
    case V4L2_STD_ATSC_8_VSB: return "V4L2_STD_ATSC_8_VSB";
    case V4L2_STD_ATSC_16_VSB: return "V4L2_STD_ATSC_16_VSB";
    case V4L2_STD_PAL: return "V4L2_STD_PAL";
    case V4L2_STD_NTSC: return "V4L2_STD_NTSC";
    default: return "not match";
    }
}

typedef struct {
	int x;
	int y;
	int size;
	int bpp;
}bmpstat;

static bmpstat is_bmp_24bpp(int fd)
{
	unsigned char buff[4];
	int ret, bpp, raster, x, y;
	bmpstat stat;
	char id[2];
	read(fd, id, 2);

	if(id[0]=='B' && id[1]=='M') {
		/* read bmp size */
		ret = lseek(fd, DEF_BMP_SIZE_OFFSET, SEEK_SET);
		perror_exit(ret == -1, "lseek");

		read(fd, buff, 4);
		x = buff[0] + (buff[1]<<8) + (buff[2]<<16) + (buff[3]<<24);
		stat.x = x;
		read(fd, buff, 4);
		y = buff[0] + (buff[1]<<8) + (buff[2]<<16) + (buff[3]<<24);
		stat.y = y;

		ret = lseek(fd, DEF_BMP_BPP_OFFSET, SEEK_SET);
		perror_exit(ret == -1, "lseek");

		read(fd, buff, 2);
		bpp = buff[0] + (buff[1]<<8);
		stat.bpp = bpp;

		if (bpp == 24) {
			stat.size = x * y * (bpp/8);
		} else {
			stat.size = 0;
			return stat;
		}

		/* seek to beginning of data */
		ret = lseek(fd, DEF_BMP_RASTER_OFFSET, SEEK_SET);
		perror_exit(ret == -1, "lseek");
		read(fd, buff, 4);
		raster = buff[0] + (buff[1]<<8) + (buff[2]<<16) + (buff[3]<<24);

		ret = lseek(fd, raster, SEEK_SET);
		perror_exit(ret == -1, "lseek");
	} else {
		stat.size = 0;
	}

	return stat;
}

static void read_bmp_24bpp(int fd, char *buffer, bmpstat stat)
{
	int i;
	int bytes = (stat.bpp/8);
	char *w_bufferp = buffer + stat.size - (stat.x * bytes);

	for (i=0; i<stat.y; i++) {
		read(fd, w_bufferp, stat.x * bytes); /* copy one line */
		w_bufferp -= stat.x * bytes; /* backoff 1 lines - x*3 */
	}
}

static void init_cap_files(void)
{
	struct stat stat;
	int fd, ret;
	unsigned int i;
	bmpstat bmpstat;

	for (i = 0; i < num_cap_frames; ++i) {
		fd = open(cap_frame_file[i], O_RDONLY, 0);
		perror_exit(fd < 0, "open");

		bmpstat = is_bmp_24bpp(fd);
		if (bmpstat.size > 0) {
			p_cap_mmap[i] = (char*) malloc(bmpstat.size);
			perror_exit(MAP_FAILED == p_cap_mmap[i], "malloc");

			read_bmp_24bpp(fd, p_cap_mmap[i], bmpstat);

			if (strcmp(cap_pixfmt,"BGR3"))
				printf("Use '--cap-pixfmt BGR3' with this bmp file.\n");

			cap_width         = bmpstat.x;
			cap_height        = bmpstat.y;
			cap_frame_size[i] = bmpstat.size;
			is_bmp = true;
		} else {
			ret = fstat(fd, &stat);
			perror_exit(ret < 0, "fstat");

			cap_frame_size[i] = stat.st_size;
			p_cap_mmap[i] = mmap(NULL, cap_frame_size[i],
					      PROT_READ | PROT_WRITE, MAP_PRIVATE,
					      fd, 0);
			perror_exit(MAP_FAILED == p_cap_mmap[i], "mmap");
		}
	}
}

static bool init_cap_dev()
{
    struct v4l2_frmsizeenum fsize = {0};
    struct v4l2_format fmt = {0};
    struct v4l2_requestbuffers reqbuf = {0};
    struct v4l2_framebuffer fb_v4l2 = {0};
    int ret;

    cap_fd = open(cap_dev_name, O_RDWR, 0);
    perror_exit(cap_fd < 0, "open");

    /* print NTSC/PAL */
    v4l2_std_id std;
    ret = ioctl(cap_fd, VIDIOC_QUERYSTD, &std);
    perror_exit(ret != 0, "VIDIOC_QUERYSTD");
    debug("  TV: %s\n", d2s_std_id(std))

    /* print sensor size */
    fsize.pixel_format = V4L2_PIX_FMT_RGB24;
    ret = ioctl(cap_fd, VIDIOC_ENUM_FRAMESIZES, &fsize);
    perror_exit(ret != 0, "VIDIOC_ENUM_FRAMESIZES");
    if (fsize.type == V4L2_FRMSIZE_TYPE_STEPWISE)
        debug("  sensor size: %dx%d\n", fsize.stepwise.step_width, fsize.stepwise.step_height)
    else
        debug("  sensor size: %dx%d\n", fsize.discrete.width, fsize.discrete.height)

    fmt.type = V4L2_BUF_TYPE_VIDEO_OVERLAY;
    fmt.fmt.pix.width = dst_width;
    fmt.fmt.pix.height = dst_height;
    fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_RGB24;
    fmt.fmt.pix.bytesperline = fmt.fmt.pix.width;
    fmt.fmt.win.w.top = dst_x;
    fmt.fmt.win.w.left = dst_y;
    fmt.fmt.win.w.width = dst_width;
    fmt.fmt.win.w.height = dst_height;
    ret = ioctl(cap_fd, VIDIOC_S_FMT, &fmt);
    perror_exit(ret != 0, "VIDIOC_S_FMT");

    /* dev->io_ctx */
    reqbuf.count = 1;
    reqbuf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    reqbuf.memory = V4L2_MEMORY_MMAP;
    ret = ioctl(cap_fd, VIDIOC_REQBUFS, &reqbuf);
    perror_exit(ret != 0, "VIDIOC_REQBUFS");

    p_capture_fb = (unsigned char*) malloc(dst_width * dst_height * 3);
	perror_exit(p_capture_fb == 0, "MALLOC");

    fb_v4l2.flags = V4L2_FBUF_FLAG_OVERLAY;
    fb_v4l2.base = (void *)p_capture_fb;
    fb_v4l2.fmt.width = dst_width;
    fb_v4l2.fmt.height = dst_height;
    fb_v4l2.fmt.pixelformat = V4L2_PIX_FMT_RGB565;
    fb_v4l2.fmt.bytesperline = dst_width * 4;
    ret = ioctl(cap_fd, VIDIOC_S_FBUF, &fb_v4l2);
    perror_exit(ret != 0, "VIDIOC_S_FBUF");

    return true;
}

static void init_vout_dev(void)
{
	struct v4l2_requestbuffers reqbuf;
	struct v4l2_capability cap;
	struct v4l2_framebuffer fbuf;
	struct v4l2_buffer buf;
	struct v4l2_control rot, flip;
	struct v4l2_format fmt;
	struct v4l2_crop crop;
	int ret;
	unsigned int i;

	vout_fd = open(vout_dev_name, O_RDWR /*| O_NONBLOCK*/, 0);
	perror_exit(vout_fd < 0, "open");

	ret = ioctl(vout_fd, VIDIOC_QUERYCAP, &cap);
	perror_exit(ret != 0, "VIDIOC_QUERYCAP");

	if (!(cap.capabilities & V4L2_CAP_VIDEO_OUTPUT_OVERLAY)) {
		fprintf(stderr, "%s does not support output overlay\n",
			vout_dev_name);
		exit(EXIT_FAILURE);
	}

	/* Setup v4l2 framebuffer */
	memzero(fbuf);
	fbuf.fmt.width = dst_width;
	fbuf.fmt.height = dst_height;
	if (enable_galpha)
		fbuf.flags |= V4L2_FBUF_FLAG_GLOBAL_ALPHA;
	else if (enable_lalpha)
		fbuf.flags |= V4L2_FBUF_FLAG_LOCAL_ALPHA;
	if (enable_ck)
		fbuf.flags |= V4L2_FBUF_FLAG_SRC_CHROMAKEY;
	fbuf.fmt.pixelformat = v4l2_fourcc(dst_pixfmt[0], dst_pixfmt[1],
					   dst_pixfmt[2], dst_pixfmt[3]);

	ret = ioctl(vout_fd, VIDIOC_S_FBUF, &fbuf);
	perror_exit(ret != 0, "VIDIOC_S_FBUF");

	/* Set format for overlay (destination) */
	memzero(crop);
	crop.type		= V4L2_BUF_TYPE_VIDEO_OUTPUT;
	crop.c.width            = dst_width;
	crop.c.height           = dst_height;
	crop.c.left             = dst_x;
	crop.c.top              = dst_y;
	ret = ioctl(vout_fd, VIDIOC_S_CROP, &crop);
	perror_exit(ret != 0, "VIDIOC_S_CROP");

	memzero(fmt);
	fmt.type		= V4L2_BUF_TYPE_VIDEO_OUTPUT_OVERLAY;
	fmt.fmt.win.field	= V4L2_FIELD_ANY;
	fmt.fmt.win.chromakey   = color_key;
	fmt.fmt.win.global_alpha = galpha;
	ret = ioctl(vout_fd, VIDIOC_S_FMT, &fmt);
	perror_exit(ret != 0, "VIDIOC_S_FMT");

	flip.id = V4L2_CID_HFLIP;
	flip.value = dst_flip_h ? 1 : 0;
	ret = ioctl(vout_fd, VIDIOC_S_CTRL, &flip);
	perror_exit(ret != 0, "VIDIOC_S_CTRL");

	flip.id = V4L2_CID_VFLIP;
	flip.value = dst_flip_v ? 1 : 0;
	ret = ioctl(vout_fd, VIDIOC_S_CTRL, &flip);
	perror_exit(ret != 0, "VIDIOC_S_CTRL");

	rot.id = V4L2_CID_ROTATE;
	rot.value = dst_rotation;
	ret = ioctl(vout_fd, VIDIOC_S_CTRL, &rot);
	perror_exit(ret != 0, "VIDIOC_S_CTRL");

	/* Set format for output (source) */
	fmt.type                = V4L2_BUF_TYPE_VIDEO_OUTPUT;
	fmt.fmt.pix.width	= cap_width;
	fmt.fmt.pix.height	= cap_height;
	fmt.fmt.pix.pixelformat = v4l2_fourcc(cap_pixfmt[0], cap_pixfmt[1],
					      cap_pixfmt[2], cap_pixfmt[3]);
	fmt.fmt.pix.field	= V4L2_FIELD_ANY;

	ret = ioctl(vout_fd, VIDIOC_S_FMT, &fmt);
	perror_exit(ret != 0, "VIDIOC_S_FMT");

	/* read width/height back in case it changed */
	ret = ioctl(vout_fd, VIDIOC_G_FMT, &fmt);
	perror_exit(ret != 0, "VIDIOC_G_FMT");
	if (cap_width != fmt.fmt.pix.width ||
	    cap_height != fmt.fmt.pix.height) {
		fprintf(stderr,
			"output cannot do %dx%d, try \"-w%d -h%d\"\n",
			cap_width, cap_height,
			fmt.fmt.pix.width, fmt.fmt.pix.height);
		exit(EXIT_FAILURE);
	}

	memzero(reqbuf);
	reqbuf.count  = num_buffers;
	reqbuf.type   = V4L2_BUF_TYPE_VIDEO_OUTPUT;
	reqbuf.memory = V4L2_MEMORY_MMAP;
	ret = ioctl(vout_fd, VIDIOC_REQBUFS, &reqbuf);
	perror_exit(ret != 0, "VIDIOC_REQBUFS");
	num_vout_frames = reqbuf.count;
	debug("Got %d output buffers\n", num_vout_frames);

	for (i = 0; i < num_vout_frames; ++i) {
		buf.type	= V4L2_BUF_TYPE_VIDEO_OUTPUT;
		buf.memory	= V4L2_MEMORY_MMAP;
		buf.index	= i;

		ret = ioctl(vout_fd, VIDIOC_QUERYBUF, &buf);
		perror_exit(ret != 0, "VIDIOC_QUERYBUF");
		debug("QUERYBUF returned offset: %x\n", buf.m.offset);

		vout_frame_size[i] = buf.length;
		p_vout_frame[i] = mmap(NULL, buf.length,
				       PROT_READ | PROT_WRITE,
				       MAP_SHARED,
				       vout_fd, buf.m.offset);
		perror_exit(MAP_FAILED == p_vout_frame[i], "mmap");
	}
}


static void start_vout(void)
{
	enum v4l2_buf_type type;
	int ret;

	type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
	ret = ioctl(vout_fd, VIDIOC_STREAMON, &type);
	debug("STREAMON output: %d\n", ret);
	perror_exit(ret != 0, "VIDIOC_STREAMON");
}


static void stop_vout(void)
{
    enum v4l2_buf_type type;
    int ret;
    int flag = 0;

    type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
    ret = ioctl(vout_fd, VIDIOC_STREAMOFF, &type);
    debug("STREAMOFF output: %d\n", ret);
    perror_exit(ret != 0, "VIDIOC_STREAMOFF");

    ret = ioctl(cap_fd, VIDIOC_OVERLAY, &flag);
    debug("capture OVERLAY output: %d\n", ret);
    perror_exit(ret != 0, "VIDIOC_OVERLAY");
}

static void start_cap(void)
{
    enum v4l2_buf_type type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    int flag = 1;
    int ret;

    ret = ioctl(cap_fd, VIDIOC_STREAMON, &type);
	debug("capture STREAMON output: %d\n", ret);
    perror_exit(ret != 0, "VIDIOC_STREAMON");

    ret = ioctl(cap_fd, VIDIOC_OVERLAY, &flag);
    debug("capture OVERLAY output: %d\n", ret);
    perror_exit(ret != 0, "VIDIOC_OVERLAY");
}

/* wait for and return a new captured frame */
static int dq_frame_cap(struct v4l2_buffer *buf)
{
	memzero(*buf);

	buf->index = 0;
	buf->bytesused = cap_frame_size[0];

	vdebug("index %d\n", buf->index);

	/* Verify we've got a correct buffer */
	assert(buf->index < num_cap_frames);

	return 0;
}


/* queue a captured frame for display */
static int q_frame_vout(struct v4l2_buffer *capbuf)
{
	struct v4l2_buffer buf;
	int ret;

	memzero(buf);
	buf.type      = V4L2_BUF_TYPE_VIDEO_OUTPUT;
	buf.memory    = V4L2_MEMORY_MMAP;
	buf.index     = capbuf->index;
	buf.timestamp.tv_sec = 1;
	buf.timestamp.tv_usec = 0;

	memcpy(p_vout_frame[buf.index], p_cap_mmap[buf.index],
	       capbuf->bytesused);

	ret = ioctl(vout_fd, VIDIOC_QBUF, &buf);
	perror_ret(ret != 0, "VIDIOC_QBUF");

	vdebug("index %d\n", buf.index);
	return 0;
}

/* wait for and return a displayed frame */
static int dq_frame_vout(struct v4l2_buffer *buf)
{
	int ret;

	memzero(*buf);
	buf->type   = V4L2_BUF_TYPE_VIDEO_OUTPUT;
	buf->memory = V4L2_MEMORY_MMAP;

	ret = ioctl(vout_fd, VIDIOC_DQBUF, buf);
	if (ret) {
		switch (errno) {
		case EAGAIN:
			debug("Got EAGAIN\n");
			return 1;

		case EIO:
			debug("Got EIO\n");
			return 1;

		default:
			perror("VIDIOC_DQBUF");
			return 1;
		}
	}
	vdebug("index %d\n", buf->index);

	/* Verify we've got a correct buffer */
	assert(buf->index < num_vout_frames);

	return 0;
}

static void usage(FILE *fp, int argc, char **argv)
{
	(void) argc;

	fprintf(fp,
		"\nThis program streams video frames to the i.MX6 Video\n"
		"Output Overlay device."
		"A splash image must be given on the command line.\n"
		"If bmp is used, make sure it is a 24bpp BGR3 file.\n"
		"The application switches from the defined splashimage "
		"to the capture device source.\n\n"

		"Usage: %s [--splash <timeout>] <image>.bmp|<raw image>"
		" [options]\n\n"

		"Options:\n"
		"-? | --help          Print this message\n"
		"-c | --capdev name   V4L2 Capture device name [%s]\n"
		"-o | --outdev name   V4L2 Output Overlay device name [%s]\n"
		"-b | --num-buffers   Number of V4L2 buffers to request for "
		"capture and output overlay [%d]\n"
		"-i | --cap-input     Capture input number [%d]\n"
		"-w | --cap-width     Width of capture frames [%d]\n"
		"-h | --cap-height    Height of capture frames [%d]\n"
		"-p | --cap-pixfmt    V4L2 fourcc pixel format of capture "
		"frames [%s]\n"
		"-C | --cap-crop      Capture crop rectangle [%dx%d@%dx%d]\n"
		"-z | --cap-fr        Framerate of capture frames [%d]\n"
		"-m | --cap-motion    Deinterlace motion compensation [%d]\n"
		"-W | --disp-width    Width of the overlay [%d]\n"
		"-H | --disp-height   Height of the overlay [%d]\n"
		"-X | --disp-x        X Position of overlay [%d]\n"
		"-Y | --disp-y        Y Position of overlay [%d]\n"
		"-P | --disp-pixfmt   V4L2 pixel format of overlay [%s]\n"
		"-f | --flip          Flip (h, H, v, V)\n"
		"-r | --cap-rot       Rotate capture frames (degrees) [%d]\n"
		"-R | --disp-rot      Rotate display frames (degrees) [%d]\n"
		"-a | --galpha        Global alpha of overlay [%d]\n"
		"-l | --lalpha        Enable local alpha of overlay [%d]\n"
		"-k | --color-key     Color Key of overlay [0x%08x]\n"
		"-S | --splash        Splashscreen timeout [%d] msec\n"
		"-T | --camera        Camera-preview timeout [%d] msec, setting 0 is infinite loop\n"
		"-K | --use-key       Splash <--> Camera switch use keyboard [%d]\n"
		"-D | --connector     Display connector name [%s]\n",
		argv[0], cap_dev_name, vout_dev_name, num_buffers,
		cap_input, cap_width, cap_height, cap_pixfmt,
		cap_crop.c.width, cap_crop.c.height,
		cap_crop.c.left, cap_crop.c.top, cap_fr, cap_motion,
		dst_width, dst_height, dst_x, dst_y, dst_pixfmt,
		cap_rotation, dst_rotation, galpha, enable_lalpha, color_key,
		splash_timeout, camera_timeout, use_key, connector_name);
}

static const char short_options[] =
	"c:C:o:b:i:w:h:p:z:m:W:H:X:Y:P:r:R:f:a:k:S:T:l:D:K?";

static const struct option long_options[] = {
	{ "help",        no_argument,       NULL, '?' },
	{ "capdev",      required_argument, NULL, 'c' },
	{ "outdev",      required_argument, NULL, 'o' },
	{ "num-buffers", required_argument, NULL, 'b' },
	{ "cap-input",   required_argument, NULL, 'i' },
	{ "cap-width",   required_argument, NULL, 'w' },
	{ "cap-height",  required_argument, NULL, 'h' },
	{ "cap-pixfmt",  required_argument, NULL, 'p' },
	{ "cap-crop",    required_argument, NULL, 'C' },
	{ "cap-fr",      required_argument, NULL, 'z' },
	{ "cap-motion",  required_argument, NULL, 'm' },
	{ "disp-width",  required_argument, NULL, 'W' },
	{ "disp-height", required_argument, NULL, 'H' },
	{ "disp-x",      required_argument, NULL, 'X' },
	{ "disp-y",      required_argument, NULL, 'Y' },
	{ "disp-pixfmt", required_argument, NULL, 'P' },
	{ "flip",        required_argument, NULL, 'f' },
	{ "cap-rot",     required_argument, NULL, 'r' },
	{ "disp-rot",    required_argument, NULL, 'R' },
	{ "galpha",      required_argument, NULL, 'a' },
	{ "lalpha",      no_argument,       NULL, 'l' },
	{ "color-key",   required_argument, NULL, 'k' },
	{ "splash",      required_argument, NULL, 'S' },
	{ "camera",      required_argument, NULL, 'T' },
	{ "use-key",     no_argument,       NULL, 'K' },
	{ "connector",   required_argument, NULL, 'D' },
	{ 0, 0, 0, 0 }
};

enum {
	CAP_CROP_WIDTH = 0,
	CAP_CROP_HEIGHT,
	CAP_CROP_X,
	CAP_CROP_Y,L
};

static const char *crop_opts[] =
{
	[CAP_CROP_WIDTH]  = "width",
	[CAP_CROP_HEIGHT] = "height",
	[CAP_CROP_X]      = "left",
	[CAP_CROP_Y]      = "top",
	NULL
};

static void parse_options(int argc, char **argv)
{
	char *subopts, *value;
	int idx, c;
	int i;

	for (;;) {
		c = getopt_long(argc, argv, short_options,
				long_options, &idx);
		if (-1 == c)
			break;

		switch (c) {
		case 0: /* getopt_long() flag */
			break;
		case 'c':
			cap_dev_name = optarg;
			break;
		case 'o':
			vout_dev_name = optarg;
			break;
		case 'b':
			num_buffers = strtol(optarg, NULL, 0);
			if (num_buffers > MAX_NUM_BUFFERS) {
				fprintf(stderr,
					"Invalid num-buffers, max is %d\n",
					MAX_NUM_BUFFERS);
				exit(EXIT_FAILURE);
			}
			break;
		case '?':
			usage(stdout, argc, argv);
			exit(EXIT_SUCCESS);
		case 'i':
			cap_input = strtol(optarg, NULL, 0);
			break;
		case 'w':
			cap_width = strtol(optarg, NULL, 0);
			break;
		case 'h':
			cap_height = strtol(optarg, NULL, 0);
			break;
		case 'p':
			cap_pixfmt = optarg;
			break;
		case 'z':
			cap_fr = strtol(optarg, NULL, 0);
			break;
		case 'm':
			cap_motion = strtol(optarg, NULL, 0);
			break;
		case 'C':
			subopts = optarg;
			while (*subopts != '\0') {
				char *saved = subopts;
				switch(getsubopt(&subopts, (char **)crop_opts,
						 &value)) {
				case CAP_CROP_WIDTH:
					if (!value)
						goto fail_usage;
					cap_crop.c.width = atoi(value);
					break;
				case CAP_CROP_HEIGHT:
					if (!value)
						goto fail_usage;
					cap_crop.c.height = atoi(value);
					break;
				case CAP_CROP_X:
					if (!value)
						goto fail_usage;
					cap_crop.c.left = atoi(value);
					break;
				case CAP_CROP_Y:
					if (!value)
						goto fail_usage;
					cap_crop.c.top = atoi(value);
					break;
				default:
					fprintf(stderr,
						"Unknown suboption `%s'\n",
						saved);
					goto fail_usage;
				}
			}
			break;
		case 'W':
			dst_width = strtol(optarg, NULL, 0);
			break;
		case 'H':
			dst_height = strtol(optarg, NULL, 0);
			break;
		case 'X':
			dst_x = strtol(optarg, NULL, 0);
			break;
		case 'Y':
			dst_y = strtol(optarg, NULL, 0);
			break;
		case 'P':
			dst_pixfmt = optarg;
			break;
		case 'r':
			cap_rotation = strtol(optarg, NULL, 0);
			break;
		case 'R':
			dst_rotation = strtol(optarg, NULL, 0);
			break;
		case 'f':
			switch (*optarg) {
			case 'h':
				cap_flip_h = true;
				break;
			case 'H':
				dst_flip_h = true;
				break;
			case 'v':
				cap_flip_v = true;
				break;
			case 'V':
				dst_flip_v = true;
				break;
			default:
				fprintf(stderr, "invalid flip `%s'\n", optarg);
				goto fail_usage;
			}
			break;
		case 'a':
			galpha = strtoul(optarg, NULL, 0);
			enable_galpha = true;
			break;
		case 'l':
			enable_lalpha = true;
			break;
		case 'k':
			color_key = strtoul(optarg, NULL, 0);
			enable_ck = true;
			break;
		case 'S':
			splash_timeout =  strtoul(optarg, NULL, 0);
			if (splash_timeout < 0) {
				fprintf(stderr, "invalid timeout `%s'\n", optarg);
				goto fail_usage;
			}
			break;
		case 'T':
			camera_timeout =  strtoul(optarg, NULL, 0);
			if (camera_timeout < 0) {
				fprintf(stderr, "invalid timeout `%s'\n", optarg);
				goto fail_usage;
			}
			break;
		case 'K':
			use_key = true;
			break;
		case 'D':
			connector_name = optarg;
			break;
		default:
			goto fail_usage;
		}
	}

	for (i = optind; i < argc && (int) num_cap_frames <= num_buffers; i++)
		cap_frame_file[num_cap_frames++] = argv[i];

	if (!num_cap_frames || num_cap_frames > 1) {
		usage(stderr, argc, argv);
		exit(EXIT_FAILURE);
	}

	/* local alpha takes precedence over global alpha */
	if (enable_lalpha)
		enable_galpha = false;

	/* millisec to microsec */
	splash_timeout *= 1000;
	camera_timeout *= 1000;

	return;

fail_usage:
	usage(stderr, argc, argv);
	exit(EXIT_FAILURE);
}

static bool no_sigint = true;

static void sigint_handler(int dummy)
{
	(void) dummy;
	no_sigint = false;
}

static void release_splash()
{
	if (is_bmp) {
		free(p_cap_mmap[0]);
		p_cap_mmap[0] = NULL;
	}
	else {
		munmap(p_cap_mmap[0], cap_frame_size[0]);
	}
		munmap(p_vout_frame[0], vout_frame_size[0]);

}

static bool show_splash(int timeout)
{
	struct v4l2_buffer capbuf;
	struct v4l2_buffer outbuf;

	start_vout();

	if (dq_frame_cap(&capbuf)) {
		fprintf(stderr, "dq_frame_cap failed\n");
		return false;
	}

	if (q_frame_vout(&capbuf)) {
		fprintf(stderr, "q_frame_vout failed\n");
		return false;
	}

	if (!use_key)
	{
		usleep (timeout);
	}

	if (dq_frame_vout(&outbuf)) {
		fprintf(stderr, "dq_frame_vout failed\n");
		return false;
	}
	return true;
}

static void key_switch()
{
    int tmp = -1;
    int switch_preview = 1;
    int ch = 0;

    while (no_sigint) {
        if (switch_preview == 1) {
            if (tmp != switch_preview){
                printf("Splash\n");
                stop_vout();
                tmp = switch_preview;
                if(!show_splash(splash_timeout))
                    return;
            }
        } else {
            if (tmp != switch_preview) {
                printf("Camera\n");
                stop_vout();
                tmp = switch_preview;
                start_cap();
            }
        }
        printf("please key : switch -> s, fin -> q\n");
        ch = getchar();
        if (!no_sigint || ch == 'q'){
            break;
        }
        if (ch == 's') {
            switch_preview *= -1;
            while(getchar() != '\n');
        } else if (ch != '\n') {
            while(getchar() != '\n');
    }
}

        printf("fin\n");
        release_splash();
}

int main(int argc, char **argv)
{
	int ret = 0;
	int fd_drm = 0;

	parse_options(argc, argv);

	signal(SIGINT, sigint_handler);

	init_cap_files();
	init_vout_dev();
	init_cap_dev();

	fd_drm = open(dri_card, O_RDWR | O_CLOEXEC);
	if (fd_drm < 0) {
		perror("cannot open drm device");
		return -1;
	}

	ret = drm_mode_set_helper_init(connector_name, fd_drm);
	if (ret < 0) {
		return -1;
	}

	if(use_key) {
		key_switch();
	}else {
		if(!show_splash(splash_timeout))
				goto fail_loop;
		release_splash();

	        stop_vout();
		start_cap();

		while (no_sigint) {
			if (camera_timeout == 0){
				sleep(1);
			}else {
				usleep(camera_timeout);
				break;
			}
		}
	}

fail_loop:
	if (cap_dev_name)
		close(cap_fd);
	close(vout_fd);

	return ret;
}
